package parser;

/**
 * Plus token class.
 */
public class PlusToken extends AToken {
    /**
     * Interface for a visitor to visit a plus token.
     */
    public static interface IPlusVisitor extends ITokVisitor {
        /**
         * Case for a plus token.
         *
         * @param host  plus token
         * @param param visitor-specific parameter
         * @return visitor-specific return value
         */
        public Object plusCase(PlusToken host, Object param);
    }

    /**
     * Abstract class for a chained visitor to visit a plus token. In all non-plus cases, the successor is called.
     */
    public static abstract class AChainVis implements IPlusVisitor {
        /**
         * Successor.
         */
        private ITokVisitor _successor;

        /**
         * Constructor.
         *
         * @param successor visitor to serve as successor in the chain of responsibility.
         */
        protected AChainVis(ITokVisitor successor) {
            _successor = successor;
        }

        /**
         * Continue in the chain of responsibility.
         *
         * @param host  token
         * @param param visitor-specific parameter
         * @return visitor-specific return value
         */
        public Object defaultCase(AToken host, Object param) {
            return host.execute(_successor, param);
        }
    }

    /**
     * Singleton instance.
     */
    public final static PlusToken Singleton = new PlusToken();

    /**
     * Private singleton constructor.
     */
    private PlusToken() {
        super("+");
    }

    /**
     * Execute the plus case of the visitor.
     *
     * @param algo  visitor
     * @param param visitor-specific parameter
     * @return visitor-specific return value
     */
    public Object execute(ITokVisitor algo, Object param) {
        return (algo instanceof IPlusVisitor) ?
               ((IPlusVisitor) algo).plusCase(this, param) :
               algo.defaultCase(this, param);
    }
}

